********************************************************************************
*PROGRAMACIÓN POBREZA MULTIDIMENSIONAL - CASEN 2022*****************************
********************************************************************************

/* Explicación de programación para la generación de indicadores y medida de 
pobreza multidimensional:

El presente archivo genera los indicadores que se utilizan para identificar
y medir carencias en las dimensiones de Educación, Salud, Trabajo y Seguridad
Social, Vivienda y Entorno y Redes y Cohesión Social. Dichas dimensiones son
agregadas para obtener el índice de pobreza multidimensional (variables 
pobreza_multi_4d y pobreza_multi_5d) para la base de datos de la Encuesta
Casen 2022 publicada en la página del Observatorio Social del Ministerio de 
Desarrollo Social y Familia.

El do file se organiza de la siguiente forma:

Primero en la línea 73 de la programación debe detallar la ubicación donde 
descargue la base de datos publicada en la página del Observatorio Social.

En el apartado "PREVIO" se excluyen los casos pertenecientes a núcleos
compuestos por el servicio doméstico (nucleo = 0) así como también se crean 
variables necesarias para poder elaborar las respectivas carencias en cada
una de las dimensiones consideradas.

"EDUCACIÓN", "SALUD", "TRABAJO Y SEGURIDAD SOCIAL" "VIVIENDA Y ENTORNO" y 
"REDES Y COHESION SOCIAL" elabora cada una de las carencias respectivas a la 
dimensión respectiva. Dentro del apartado se encuentran las definiciones de 
población de referencia, así como también la carencia a nivel poblacional.

"AGREGACIÓN A NIVEL DE HOGARES" genera las variables agregadas a nivel de hogar
para cada carencia, según sus poblaciones de referencia.

"INDICADOR SINTÉTICO" se compone de 2 etapas. Por un lado, el proceso de 
identificación en donde se determinan cuáles hogares clasifican en situación
de pobreza multidimensional. Primero, se pondera a cada una de las 
carencias, de acuerdo a la actual metodología establecida para Chile se 
considera que cada dimensión tradicional: Educación, Salud, Trabajo y Seguridad 
Social, Vivienda y Entorno; tienen la misma ponderación (22.5%). Y en cuanto a 
la dimensión de Redes y Cohesión Social tiene un peso de 10%. Dentro de cada 
dimensión, los indicadores tienen igual peso. Posteriormente se establece el 
vector de conteo, que corresponde a la suma ponderada de cada indicador. 
Además, se identifican los hogares que no cuentan con información en alguno los 
indicadores que componen la medida. Una vez determinado el vector de conteo se 
clasifica la situación de pobreza para los hogares que disponen información 
completa. El umbral para determinar una situación de pobreza es de k = 22,5%, 
calculando así el indicador de Incidencia de Pobreza Multidimensional (“h_k”).

En la segunda etapa, la agregación, se crean el indicador de intensidad de 
la pobreza multidimensional (“a_k”), el cual establece el promedio ponderado de 
carencias que experimentan los hogares en situación de pobreza. Asimismo, en 
esta etapa se elabora la tasa de recuento ajustada “m_k”, que corresponde la 
multiplicación de la incidencia (h_k) por la intensidad de la pobreza (a_k).

Después se realiza el mismo proceso para estimar la incidencia de la pobreza
medida con cuatro dimensiones: Educación, Salud, Trabajo y Seguridad Social y 
Vivienda (todas con una ponderación de 25% y el umbral para determinar la 
situación de pobreza multidimensional es de k = 25%). Se genera la incidencia 
(h_ant), intensidad (a_ant) y tasa de recuento ajustada (m_ant) de la medida
de 4 dimensiones. 

El objetivo de este proceso de la programación es establecer para este año 
comparabilidad con las estimaciones presentadas en el año 2009, 2011, 
2013, 2015 y 2017.

Finalmente, a modo de simplificar, las incidencias de ambas medidas son 
renombradas como "pobreza_multi_5d" y "pobreza_multi_4d" respectivamente.
*/

clear all
set more off, permanently

use "BASEDATOS.dta" //Indicar ruta de la base de datos pública.


********************************************************************************
********************************PREVIO******************************************
{/*****************************************************************************/

*Excluimos el núcleo 0:
drop if nucleo == 0


rename area zona
label define zona 1 "Urbano" 2 "Rural"
label values zona zona

*Variable asistencia
capture drop asiste
gen asiste=.
replace asiste=1 if (e3==1)
replace asiste=2 if (e3==2)

*Escolaridad
capture drop esc
gen esc=.
replace esc=0 if      (edad>= 15 & e6a==1)
replace esc=0 if      (edad>= 15 & (e6a==2|e6a==3|e6a==4|e6a==5))
replace esc=e6b if    (edad >= 15 & (e6a==6|e6a==7)) & e6b != -88
replace esc=e6b+6 if  (edad>= 15 & (e6a==8|e6a==10)) & e6b != -88
replace esc=e6b+8 if  (edad>= 15 & (e6a==9|e6a==11)) & e6b != -88
replace esc=e6b+12 if (edad>=15 & (e6a>=12 & e6a<=13)) & e6b != -88
replace esc=e6b+12+e6d_preg if (edad>=15 & (e6a == 14)) & e6b != -88 & e6d_preg != -88
replace esc=e6b+12+e6d_preg+e6d_postg if (edad>=15 & (e6a == 15)) & e6b != -88 & e6d_preg != -88 & e6d_postg != -88
replace esc=. if (e6a==-88|e6b==-88)
replace esc=. if (e6a==-88|e6b==-88|e6d_preg==-88) & e6b == 14
replace esc=. if (e6a==-88|e6b==-88|e6d_preg==-88|e6d_postg==-88) & e6b == 15

*Escolaridad considerando a menores de 15 años
capture drop esc_2
gen esc_2=esc
replace esc_2=0 if (esc==. & e6a<=5) & e6a!=-88
replace esc_2=e6b if (esc==. & e6a==7) & e6a!=-88
replace esc_2=e6b+8 if (esc==. & (e6a==9|e6a==11)) & e6a!=-88
replace esc_2=. if (e6a==-88|e6b==-88) & esc==.

*Generamos curso y grado para carencia de rezago
capture drop grado
gen grado=.
replace grado=1 if e6b_asiste==1 & e6a_asiste==7 & e6a_asiste!=-88 & e6b_asiste!=-88
replace grado=2 if e6b_asiste==2 & e6a_asiste==7 & e6a_asiste!=-88 & e6b_asiste!=-88
replace grado=3 if e6b_asiste==3 & e6a_asiste==7 & e6a_asiste!=-88 & e6b_asiste!=-88
replace grado=4 if e6b_asiste==4 & e6a_asiste==7 & e6a_asiste!=-88 & e6b_asiste!=-88
replace grado=5 if e6b_asiste==5 & e6a_asiste==7 & e6a_asiste!=-88 & e6b_asiste!=-88
replace grado=6 if e6b_asiste==6 & e6a_asiste==7 & e6a_asiste!=-88 & e6b_asiste!=-88
replace grado=7 if e6b_asiste==7 & e6a_asiste==7 & e6a_asiste!=-88 & e6b_asiste!=-88
replace grado=8 if e6b_asiste==8 & e6a_asiste==7 & e6a_asiste!=-88 & e6b_asiste!=-88
replace grado=9 if e6b_asiste==1 & (e6a_asiste==9|e6a_asiste==11) & e6a_asiste!=-88 & e6b_asiste!=-88
replace grado=10 if e6b_asiste==2 & (e6a_asiste==9|e6a_asiste==11) & e6a_asiste!=-88 & e6b_asiste! =-88
replace grado=11 if e6b_asiste==3 & (e6a_asiste==9|e6a_asiste==11) & e6a_asiste!=-88 & e6b_asiste!=-88
replace grado=12 if (e6b_asiste==4|e6b_asiste==5) & (e6a_asiste==9|e6a_asiste==11) & e6a_asiste!=-88 & e6b_asiste!=-88

capture drop curso
gen curso=grado+6 if asiste==1

}


********************************************************************************
******************************EDUCACIÓN*****************************************
{/*****************************************************************************/

********************************************************************************
/* Indicador 1: Asistencia a un establecimiento educacional

La población de referencia corresponde a las personas entre 4 y 18, 
incluyendo también a quienes tienen entre 6 y 26 años encontrándose en 
situación de discapacidad.

Se considera carente a los hogares en que al menos uno de sus integrantes de la 
población de referencia no asiste a un establecimiento educacional.

Se excluyen de los carentes a las personas que han culminado la enseñanza 
media, es decir, cuentan con los 12 años de educación obligatoria.
*/
********************************************************************************

*Población de referencia:
gen pob_asis=.
replace pob_asis=0 if (edad<4 | edad>18) & edad!=.
replace pob_asis=1 if (edad>=4 & edad<=18) & edad!=.
replace pob_asis=1 if (edad>=6 & edad<=26) & (s31_1 == 1 | s31_2 == 1 | s31_3 == 1 | s31_4 == 1 | s31_5 == 1 | s31_6 == 1)
replace pob_asis=0 if (edad<6) & (s31_1 == 1 | s31_2 == 1 | s31_3 == 1 | s31_4 == 1 | s31_5 == 1 | s31_6 == 1)
replace pob_asis=0 if (esc_2>=12 & esc_2!=.)
replace pob_asis=8 if esc_2==. & pob_asis==1 & edad>=15
replace pob_asis=8 if ((edad>=4 & edad<6)|(edad>18 & edad<=26)) & (s31_1 == -88 | s31_2 == -88 | s31_3 == -88 | s31_4 == -88 | s31_5 == -88 | s31_6 == -88) &  esc_2<12
replace pob_asis=8 if pob_asis==.
	
* Privación a nivel individual
gen d_asis=.
replace d_asis=1 if (asiste==2) & pob_asis==1
replace d_asis=0 if (asiste==1) & pob_asis==1
replace d_asis=9 if d_asis==. & pob_asis==1 


********************************************************************************
/* Indicador 2: Rezago Escolar

La población de referencia incluye a toda persona de 21 años o menos que 
asiste a educación básica, educación media científico-humanista o educación 
media técnico profesional.

Se considera carentes a aquellos hogares en que al menos uno de sus 
integrantes de la población de referencia se encuentra retrasado en dos 
o más años en sus estudios.
*/
********************************************************************************


*Población de referencia
gen pob_rez=.
replace pob_rez=0 if edad>21 & edad!=.
replace pob_rez=0 if asiste==2
replace pob_rez=0 if (e6a_asiste!=7 & e6a_asiste!=9 & e6a_asiste!=11)

replace pob_rez=1 if (asiste==1 & (e6a_asiste==7|e6a_asiste==9|e6a_asiste==11) & (edad<=21) & (e6a_asiste!=-88))
replace pob_rez=0 if esc_2>=12 & esc_2!=. & pob_rez==1
	
* Privación a nivel individual
gen d_rez=.
replace d_rez=1 if(edad-curso>=2) & pob_rez==1 & asiste!=. & esc_2!=. & edad!=.
replace d_rez=0 if(edad-curso<2) & pob_rez==1 & asiste!=. & esc_2!=. & edad!=.
replace d_rez=9 if d_rez==. & pob_rez==1



********************************************************************************
/* Indicador 3: Escolaridad

La población de referencia corresponde a todas las personas mayores de 
18 años, tanto quienes asisten y quienes no lo hacen.

Se considera carente a aquellos hogares que tienen entre sus integrantes a una
persona que ha alcanzado menos años de estudios que los establecidos por ley 
(de acuerdo a su edad)

*/
********************************************************************************
/*
Se definen exigencias por ley para cada cohorte que se ve afectada por los 
cambios legislativos:

a. 1920 a 1929: 4 años
b. 1930 a 1965: 6 años
c. 1966 a 2002: 8 años
d. 2003 en adelante: 12 años
*/

*Se generan los tramos etarios, considerando la edad que se debiera tener al entrar en vigencia el cambio de legislación
gen tedad=1 if (edad>112) & edad!=.

*A quienes les comenzó a regir los 4 años de escolaridad obligatoria en el 2022 tienen 112 años (1920)
replace tedad=2 if (edad>104 & edad<=112)

*A quienes les comenzó a regir los 6 años de escolaridad obligatoria en el 2022 tienen 104 años (1930)
replace tedad=3 if (edad>69 & edad<=104)

*A quienes les comenzó a regir los 8 años de escolaridad obligatoria en el 2022 tienen 69 años (1967)
replace tedad=4 if (edad>37 & edad<=69)

*A quienes les comenzó a regir los 12 años de escolaridad obligatoria en el 2022 tienen 37 años (2003)
replace tedad=5 if (edad>18 & edad<=37)


* Población de referencia
gen pob_esc=.
replace pob_esc=1 if (edad>18) & edad!=.
replace pob_esc=0 if (edad<=18) & edad!=.
replace pob_esc=8 if pob_esc==.
	
* Privación a nivel individual
gen d_esc=.
replace d_esc=0 if ((tedad==1)|((tedad==2) & esc>=4)|(tedad==3 & esc>=6)|(tedad==4 & esc>=8)|(tedad==5 & esc>=12 )) & esc!=. & pob_esc==1
replace d_esc=1 if ((tedad==2 & esc<4)|(tedad==3 & esc<6)|(tedad==4 & esc<8)|(tedad==5 & esc<12 )) & esc!=. & pob_esc==1
replace d_esc=9 if d_esc==. & pob_esc==1

}


********************************************************************************
********************************SALUD*******************************************
{/*****************************************************************************/

********************************************************************************
/* Indicador 4: Malnutrición en niños y niñas

La población de referencia considera a los niños y niñas de 0 a 6 años

Se considera carentes a los hogares que cuentan con al menos un niño o niña 
que este desnutrido, en riesgo de desnutrición, con sobrepeso u obesidad.
*/
********************************************************************************
* Población de referencia
gen pob_des= (edad<=6)
gen pob_sob= pob_des
gen pob_mal= pob_des

* Privación a nivel individual

* Para niños y niñas desnutridos o en riesgo de desnutrición
gen d_des=.
replace d_des=1 if(s2==1) & pob_des==1
replace d_des=0 if(s2==2|s2==3|s2==4) & pob_des==1
replace d_des = 9 if s2 == -88 & pob_des == 1


* Para niños y niñas con sobrepeso u obesidad
gen d_sob=.
replace d_sob=1 if (s2==3|s2==4) & pob_sob==1
replace d_sob=0 if (s2==1|s2==2) & pob_sob==1
replace d_sob = 9 if s2 == -88 & pob_sob == 1

	
* Privación a nivel individual
gen d_mal=(d_des==1|d_sob==1) if (d_des!=.|d_sob!=.) & pob_des==1
replace d_mal=9 if s2==-88 & pob_des==1


********************************************************************************
/* Indicador 5: Adscripción al Sistema de Salud

La población de referencia corresponde a todas las personas

Se considera carentes a los hogares en que al menos uno de sus integrantes 
que no este afiliado a un sistema previsional de salud y no tiene otro 
seguro complementario.
*/
********************************************************************************

*Población de referencia:
gen pob_prevs = 1
replace pob_prevs = 0 if s13 == .

*Privación a nivel individual
gen d_prevs = 0 if pob_prevs == 1
replace d_prevs = 1 if s13 == 4
replace d_prevs = 9 if s13 == -88

bys folio nucleo: egen auxi1 = max(s15)

replace d_prevs = 0 if auxi1 == 1 & pob_prevs == 1
drop auxi1


********************************************************************************
/* Indicador 6: Atención de Salud

La población de referencia corresponde a todas las personas que necesitaron 
atención médica o están en tratamiento por patología garantizada (AUGE-GES)

Se considera carentes a los hogares que cuentan con al menos un integrante que 
no recibió atención de salud en los últimos tres meses o no tuvo cobertura del 
sistema AUGE-GES por razones ajenas a su voluntad o preferencia.
*/
********************************************************************************

* Población de referencia
* Población que no tuvo consulta por enfermedad en 3 meses
gen pob_acc_a=(s17==2)
replace pob_acc_a=8 if(s17==-88)

* Población en tratamiento de enfermedades auge sin tratamiento por AUGE
gen pob_acc_b=(s29==2)
replace pob_acc_b=8 if(s29==-88)

* Población de referencia total
gen pob_acc=(pob_acc_a==1|pob_acc_b==1)
replace pob_acc=8 if (pob_acc_a==8 | pob_acc_b==8) & pob_acc==0

* Privación a nivel individual
* Privación en consultas por enfermedad
gen d_acc_a = 0 if pob_acc_a == 1
replace d_acc_a = 1 if inlist(s18,8,9,10,11,14,15) & s17 == 2
replace d_acc_a=9 if s18==-88 & pob_acc_a==1

* Privación en tratamiento de enfermedades AUGE-GES
gen d_acc_b= 0 if pob_acc_b == 1
replace d_acc_b = 1 if inlist(s30,2,3,5,6,7,8,10,11,12) & s29 == 2
replace d_acc_b=9 if s30==-88 & pob_acc_b==1
	
* Privación en atención
gen d_acc=(d_acc_a==1|d_acc_b==1) if pob_acc==1
replace d_acc=9 if (d_acc_b==9 & s17==.)|(d_acc_b==9 & d_acc_a==9)|(d_acc_a==9 & s30==.)& pob_acc==1


}


********************************************************************************
**********************TRABAJO Y SEGURIDAD SOCIAL********************************
{/*****************************************************************************/

********************************************************************************
/* Indicador 7: Ocupación

La población de referencia corresponde a las personas mayores de 18 (o menor de 
19 si ha completado la enseñanza media)

Se considera carente a los hogares que al menos uno de sus integrantes se 
encuentra desocupado.

*/
********************************************************************************

*Población de referencia
gen pob_act=.
replace pob_act=1 if edad>18
replace pob_act=0 if edad<=18
*Identificamos a quienes completaron educación media
replace pob_act = 1 if ((inlist(e6a,12,13,14,15)) | (e6a_no_asiste == 9 & e6b_no_asiste == 4) | (e6a_no_asiste == 11 & e6b_no_asiste == 5)) & inrange(edad,15,18)
*dejamos fuera a quienes no tienen información en escolaridad
replace pob_act=8 if esc==. & (edad>14 & edad<=18)

* Privación a nivel individual
gen d_act=.
replace d_act=1 if (activ==2) & pob_act==1
replace d_act=0 if (activ==1 | activ==3) & pob_act==1
replace d_act=9 if (activ==.) & pob_act==1


********************************************************************************
/* Indicador 8: Seguridad Social

La población de referencia corresponde a las personas de 15 años o más que se 
encuentren ocupadas. Se excluye a trabajadores/as independientes con educación 
superior completa (de la carencia no de la población de referencia)


Se considera carente a los hogares que al menos uno de sus integrantes no 
cotiza en el sistema previsional.

*/
********************************************************************************

* Población de referencia

*Se identifican a los independientes con educación superior completa
gen edcom = 0 
replace edcom = 1 if (inlist(e6a,14,15)) | (e6a_no_asiste == 13 & e6c_completo == 1)
gen indsup= 1 if edcom == 1 & inlist(o15,1,2) 

*Se genera a la población de referencia
gen pob_cot = 0
replace pob_cot = 1 if edad >= 15 & activ == 1



*Privación a nivel individual
gen d_cot = 0 if pob_cot == 1
*Indica que no cotiza
replace d_cot = 1 if o31 == 2 & pob_cot == 1

*Indica que cotiza pero en o32 indica que no está cotizando
replace d_cot = 1 if o31 == 1 & o32 == 6 & pob_cot == 1

*Dejamos en 0 a los independientes con educacion sup completa
replace d_cot = 0 if indsup == 1 & d_cot == 1 & pob_cot == 1

*Missing en o32 (de los que cotizan)
replace d_cot = 9 if o31 == 1 & (o32 == . | o32 == -88) & pob_cot == 1
*Missing en o31
replace d_cot = 9 if (o31 == . | o31 == -88) & pob_cot == 1


********************************************************************************
/* Indicador 9: Jubilación

La población de referencia son las personas en edad de jubilar. 

Se considera carente a los hogares que al menos uno de sus integrantes no 
recibe pensión contributiva o no contributiva o que no reciben ingreso por 
arriendos, retiro de utilidades, dividendos o intereses. 

*/
********************************************************************************
*Población de referencia
gen pob_jub = .
replace pob_jub = 1 if edad > 64 & sexo == 1
replace pob_jub = 0 if edad <= 64 & sexo == 1
replace pob_jub = 1 if edad >59 & sexo == 2
replace pob_jub = 0 if edad <=59 & sexo == 2

replace pob_jub = 9 if pob_jub==.


*Ingresos por depósitos (yah1)
*Ingresos por por acciones o bonos financieros (yah2)
*Ingresos por retiro de utilidades (yrut)
*Ingresos por arriendo de propiedades urbanas (yre1)
*Ingresos por arriendo de maquinarias y animales (yama)
*Ingresos por arriendo de propiedades agricolas (yre2)
*Ingresos por arriendo de propiedades por temporadas (yre3)

*Suma de las corrientes mencionadas:
egen yot=rowtotal(yah1 yah2 yrut yre1 yama yre2 yre3)

*Pensión de vejez con APS (y280201c):
egen ypaps=rowtotal(y280201c)

/*Pensión de vejez con PGU:
	Pensión de vejez (y2803c)
	Pensión de vejez con PGU (y280301c)
	Solo PGU (y280101)
*/

egen ypenv = rowtotal(y2803c y280301c y280101)

/*Ingresos por Montepío:
	Montepío o pensión de Viudez (ymon02)
	Montepío con PGU (ymon0101)
*/

egen ymon_f = rowtotal(ymon02 ymon0101)

/*Pensión por leyes especiales:
	Pensión por leyes especiales (yesp)
	Pensión por leyes especiales con PGU (yesp0101)
*/

egen yesp_f = rowtotal(yesp yesp0101)



/*Ingresos por jubilaciones
	Pensión de vejez (ypenv)
	Pensión de vejez con APS (ypaps)
	Ingresos por Montepío (ymon_f)
	Otras pensiones (yotp)
*/

egen yj=rowtotal(ypaps ypenv ymon_f yotp)

*Carencia
gen d_jub=.
replace d_jub=0 if pob_jub==1
replace d_jub=1 if yj==0 & pob_jub==1 

*Quienes reciben por leyes especiales de reparación son no carentes
replace d_jub=0 if yesp_f > 0 & yesp_f != . & d_jub==1 & yesp_f != .m

*Quienes reciben ingresos de la propiedad son no carentes
replace d_jub=0 if yot > 0 & yot !=. & d_jub==1 & yot != .m

}


********************************************************************************
**************************VIVIENDA Y ENTORNO************************************
{/*****************************************************************************/

********************************************************************************
/* Indicador 10: Habitabilidad

La población de referencia corresponde a todos los hogares. Un hogar es carente 
si se encuentra hacinado (2,5 ó más personas por dormitorio de uso exclusivo) o 
Reside en una vivienda precaria o en una vivienda con muros, techos y/o piso en 
mal estado.

*/
********************************************************************************

********************
/* Indicador 10A: Hacinamiento

- La población de referencia corresponde a todos los hogares
- Se consideran carentes a los hogares que se encuentran hacinado (2,5 ó más 
personas por dormitorio de uso exclusivo).

*/
********************

*Población de Referencia
gen pob_hacina = 1 if numper != .
replace pob_hacina=8 if numper==.

*Generamos variable de hacinamiento
capture drop hacina
gen 	hacina=.
replace hacina=numper/v27a if (p10 == 1 | p9 == 1) & (v27a>0) & (v27a<99) & pob_hacina == 1
replace hacina=numper/v29a if (p10 == 2)  & (v29a>0) & (v29a<99) & pob_hacina == 1

*Generamos la carencia
gen d_hacina = .
replace d_hacina = 0 if hacina < 2.5 & pob_hacina == 1 & hacina != .
replace d_hacina = 1 if hacina >= 2.5 & pob_hacina == 1 & hacina != .

*Reemplazamos los que no tienen habitaciones en la vivienda u hogar dependiendo del caso 
replace d_hacina = 1 if ((p10 == 1 | p9 == 1) & v27a == 0) | (p10 == 2 & v29a == 0) & pob_hacina==1

*Ponemos en missing los NS/NR
replace d_hacina = 9 if (v27a == -88 | v29a == -88) & d_hacina == .

********************
/* Indicador 10B: Estado de la Vivienda

La población de referencia corresponde a todos los hogares
Se consideran carentes a los hogares que residen en una vivienda precaria 
o en una vivienda con muros, techos y/o piso en mal estado.

*/
********************

* Población de Referencia
gen pob_estado=(numper!=.)
replace pob_estado=8 if numper==.

* Privación a nivel individual
gen d_estado=.
replace d_estado=1 if (v1==7|v1==10) & pob_estado==1
replace d_estado=1 if v1<=6 & (v3==3|v5==3|v7==3) & pob_estado==1
replace d_estado=1 if inlist(v1,8,9) & (v3==3|v5==3|v7==3) & pob_estado==1
replace d_estado=0 if v1<=6 & (inlist(v3,1,2) & inlist(v5,1,2) & inlist(v7,1,2)) & pob_estado==1
replace d_estado=0 if inlist(v1,8,9) & (inlist(v3,1,2) & inlist(v5,1,2) & inlist(v7,1,2)) & pob_estado==1
replace d_estado=9 if d_estado==. & pob_estado==1


********************
/* 
Cálculo de carencia
*/
********************

* Población de referencia
gen pob_habitab=(numper!=.)
replace pob_habitab=8 if numper==.

* Privación a nivel individual
gen d_habitab=.
replace d_habitab=1 if (d_hacina==1 | d_estado==1) & pob_habitab==1
replace d_habitab=0 if (d_hacina==0 & d_estado==0) & pob_habitab==1
replace d_habitab=9 if d_habitab==. & pob_habitab==1



********************************************************************************
/* Indicador 11: Servicios Básicos

La población de referencia corresponde a todos los hogares

Se consideran carentes a los hogares que residen en una vivienda sin servicios 
sanitarios básicos (WC, llave dentro de la vivienda y agua) según estándar 
urbano o rural.

*/
********************************************************************************

*Población de referencia de todos los hogares
gen pob_servbas = 1 if numper != .
gen pob_elim_excretas = pob_servbas
gen pob_agua = pob_servbas
gen pob_dis_agua = pob_servbas

*Generamos cada carencia de servicio deficitario

*Acceso a agua
capture drop agua
gen agua = .
replace agua = 0 if zona == 1 & v20 == 1 & pob_servbas == 1
replace agua = 0 if zona == 2 & inlist(v20,1,4,5,6) & pob_servbas == 1
replace agua = 1 if zona == 1 & inlist(v20,4,5,6,7) & pob_servbas == 1
replace agua = 1 if zona == 2 & inlist(v20,7) & pob_servbas == 1
replace agua = 9 if agua == .

*Distribución de agua
capture drop dis_agua
gen dis_agua = .
replace dis_agua = 0 if v22 == 1 & pob_servbas == 1
replace dis_agua = 1 if inlist(v22,2,3) & pob_servbas == 1

*Sistema de eliminación de excretas adecuado
capture drop elim_excretas
gen elim_excretas = .
replace elim_excretas = 0 if inlist(v23_sistema,1,2) & pob_servbas == 1
replace elim_excretas = 1 if inlist(v23_sistema,3,4,7) & pob_servbas == 1
replace elim_excretas = 1 if v23 == 2 & pob_servbas == 1

*Generamos la carencia
gen d_servbas = 0 if pob_servbas == 1

*Si no sabemos de una de las 3 es missing en lugar de decir que no es carente
replace d_servbas = 9 if (inlist(agua,9,.) | inlist(dis_agua,9,.) | inlist(elim_excretas,9,.))
*Si tiene alguna de las tres (independiente que las otras sean missing) entonces es carente
replace d_servbas = 1 if agua == 1 | dis_agua == 1 | elim_excretas == 1


********************************************************************************
/* Indicador 12: Entorno

La población de referencia corresponde a todos los hogares.

Un hogar se considera carente si su entorno no dispone de al menos un 
equipamiento básico y viven lejos de su lugar de trabajo (1 hora de tiempo 
de traslado).

También se considerarán carentes a aquellos hogares que residan en un entorno 
con 2 problemas de contaminación (Contaminación del aire, de ríos, de agua de 
la red pública o basura)

*/
********************************************************************************


********************
/* EQUIPAMIENTO: 

No dispone de alguno de estos tres equipamientos mínimos: 

a. Transporte público.
b. Centro educacional. 
c. Centro de salud

*/
********************

*Población de referencia
gen pob_equipo=(numper!=.)
replace pob_equipo=8 if numper==.

*Servicio de transporte público
gen d_eqtr=.
replace d_eqtr=1 if v35a==2
replace d_eqtr=0 if v35a==1
replace d_eqtr=. if inlist(v35a,-88,88)

*Servicio de centro educacional
gen d_eqce=.
replace d_eqce=1 if v35b==2
replace d_eqce=0 if v35b==1
replace d_eqce=. if inlist(v35b,-88,88)

*Servicio de centro de salud
gen d_eqcs=.
replace d_eqcs=1 if v35c==2
replace d_eqcs=0 if v35c==1
replace d_eqcs=. if inlist(v35c,-88,88)

*Acceso a equipos
gen d_equipo=.
replace d_equipo=0 if (d_eqtr==0 & d_eqce==0 & d_eqcs==0) & pob_equipo==1
replace d_equipo=1 if (d_eqtr==1 | d_eqce==1 | d_eqcs==1) & pob_equipo==1
replace d_equipo=9 if d_equipo==. & pob_equipo==1


********************
/* TIEMPO DE TRASLADO: 

Tiempo de traslado al trabajo en transporte público o particular 
no motorizado igual o mayor a 60 min.

*/
********************

* Población de referencia
g pob_tiempo=(activ==1)
replace pob_tiempo=8 if activ==.

* Privación a nivel individual
gen d_tiempo=0 if (o28a_hr<1 & o28a_hr!=-8 & o28a_hr!=.) & pob_tiempo==1
replace d_tiempo=1 if (o28a_hr>=1 & o28a_hr!=-8 & o28a_hr!=.) & pob_tiempo==1
replace d_tiempo=. if o28a_hr==. & pob_tiempo==1
replace d_tiempo=. if o28a_hr==-8 & pob_tiempo==1

*Medio de transporte
replace d_tiempo=0 if (o28c==2 | o28c==6) & d_tiempo==1 & pob_tiempo==1
replace d_tiempo=. if (o28c==.) & d_tiempo==1 & pob_tiempo==1
replace d_tiempo=. if (o28c==-88) & d_tiempo==1 & pob_tiempo==1

*CARENCIA EN TIEMPO DE TRASLADO POR HOGAR
gen oc = (activ==1)
bys folio: egen hh_oc=max(oc)
bys folio: egen d_hh_tiempo=max(d_tiempo) if hh_oc==1



********************
/*
Sub-carencia de accesibilidad:
*/
********************

gen pob_accesi=1
gen d_accesi=.
replace d_accesi=1 if (d_hh_tiempo==1 & d_equipo==1) & hh_oc==1
replace d_accesi=0 if (d_hh_tiempo==0 | d_equipo==0) & hh_oc==1
replace d_accesi=1 if (d_equipo==1) & hh_oc==0
replace d_accesi=0 if (d_equipo==0) & hh_oc==0
replace d_accesi=9 if d_accesi==.



********************
/* MEDIOAMBIENTE: 

Durante los últimos 12 meses siempre presencia a lo menos DOS de las siguientes 
situaciones, : contaminación del aire, ríos, agua pública o basura

*/
********************

*Situaciones de contaminación:

*Aire
gen cont_aire=.
replace cont_aire=1 if v37b==4
replace cont_aire=0 if (v37b==1 | v37b==2 | v37b==3)
label variable cont_aire "Contaminación del aire"

*Contaminación de ríos, canales
gen cont_rios=.
replace cont_rios=1 if v37c==4
replace cont_rios=0 if (v37c==1 | v37c==2 | v37c==3)
label variable cont_rios "Contaminación de ríos, canales, esteros, lagos, tranques y embalses"

*Contaminación de agua de red pública
gen cont_agua=.
replace cont_agua=1 if v37d==4
replace cont_agua=0 if (v37d==1 | v37d==2 | v37d==3)
label variable cont_agua "Contaminación de agua de red pública"

*Contaminación por basura
gen cont_basura=.
replace cont_basura=1 if v37f==4
replace cont_basura=0 if (v37f==1 | v37f==2 | v37f==3)
label variable cont_basura "Acumulación de basura en calles, caminos, veredas o espacios públicos"
label define contaminacion 0 "Nunca, Pocas o Muchas veces" 1 "Siempre"

label values cont_aire contaminacion
label values cont_rios contaminacion
label values cont_agua contaminacion
label values cont_basura contaminacion

* Población de Referencia
gen pob_medio=(numper!=.)
replace pob_medio=8 if numper==.
*Suma de situaciones de contaminación
egen sum_con=rowtotal(cont_aire cont_rios cont_agua cont_basura)

* Privación a nivel individual
gen d_medio=.
replace d_medio=1 if (sum_con>=2) & pob_medio==1
replace d_medio=0 if (sum_con<2) & pob_medio==1
replace d_medio=9 if cont_aire==. & cont_rios==. & cont_agua==. & cont_basura==.

drop sum_con



********************
/* 
Cálculo de carencia
*/
********************

* Población de referencia
gen pob_entorno=(numper!=.)
replace pob_entorno=8 if (numper==.)
* Privación a nivel individual
gen d_entorno=.
replace d_entorno=0 if d_medio==0 & d_accesi==0 & pob_entorno==1
replace d_entorno=1 if d_medio==1 | d_accesi==1 & pob_entorno==1
replace d_entorno=9 if d_entorno==.


}


********************************************************************************
*********************REDES Y COHESIÓN SOCIAL************************************
{/*****************************************************************************/

********************************************************************************
/* Indicador 13: Apoyo y Participación Social

Un hogar es carente si no dispone de alguna persona fuera del hogar que pueda 
prestar apoyo para 8 situaciones relevantes, ninguno de sus miembros de 14 y 
más años participa en alguna organización social o grupo organizado y ninguno 
de sus miembros de 18 y más años que se encuentran ocupados está afiliado a 
alguna organización sindical, gremial o profesional relacionada con su trabajo

*/
********************************************************************************

********************
/* 
Apoyo
*/
********************
gen pob_hapoyo=(numper!=.)
replace pob_hapoyo=8 if numper==.

* Se identifican las categorías en las cuales se considera carente para cada situación

*Cuidado en caso de enfermedad en algun miembro del hogar
gen apoyo_a=.
replace apoyo_a=0 if r7a==4 
replace apoyo_a=0 if r7a==2
replace apoyo_a=1 if r7a==1
replace apoyo_a=1 if r7a==3

*Facilitar vehículo
gen apoyo_c=.
replace apoyo_c=0 if r7c==4
replace apoyo_c=0 if r7c==2
replace apoyo_c=1 if r7c==1
replace apoyo_c=1 if r7c==3

*Prestar dinero
gen apoyo_d=.
replace apoyo_d=0 if r7d==4
replace apoyo_d=0 if r7d==2
replace apoyo_d=1 if r7d==1
replace apoyo_d=1 if r7d==3

*Hacer trámites
gen apoyo_e=.
replace apoyo_e=0 if r7e==4
replace apoyo_e=0 if r7e==2
replace apoyo_e=1 if r7e==1
replace apoyo_e=1 if r7e==3

*Ayudar en uso de tecnologías
gen apoyo_f=.
replace apoyo_f=0 if r7f==4
replace apoyo_f=0 if r7f==2
replace apoyo_f=1 if r7f==1
replace apoyo_f=1 if r7f==3

*Reparaciones en el hogar
gen apoyo_g=.
replace apoyo_g=0 if r7g==4
replace apoyo_g=0 if r7g==2
replace apoyo_g=1 if r7g==1
replace apoyo_g=1 if r7g==3

*Conseguir trabajo
gen apoyo_h=.
replace apoyo_h=0 if r7h==4
replace apoyo_h=0 if r7h==2
replace apoyo_h=1 if r7h==1
replace apoyo_h=1 if r7h==3

*Aconsejar
gen apoyo_i=.
replace apoyo_i=0 if r7i==4
replace apoyo_i=0 if r7i==2
replace apoyo_i=1 if r7i==1
replace apoyo_i=1 if r7i==3

label define apoyo_lbl_mds 0 "No conoce fuera del hogar" 1 "Conoce fuera del hogar"
label value apoyo_a apoyo_lbl_mds

label value apoyo_c apoyo_lbl_mds
label value apoyo_d apoyo_lbl_mds
label value apoyo_e apoyo_lbl_mds
label value apoyo_f apoyo_lbl_mds
label value apoyo_g apoyo_lbl_mds
label value apoyo_h apoyo_lbl_mds
label value apoyo_i apoyo_lbl_mds
label variable apoyo_a "Enfermedad"
label variable apoyo_c "Vehiculo"
label variable apoyo_d "Dinero"
label variable apoyo_e "Trámites"
label variable apoyo_f "Tecnologías"
label variable apoyo_g "Reparaciones"
label variable apoyo_h "Trabajo"
label variable apoyo_i "Problemas personales"

*** Agregación por hogar
local apoyo a c d e f g h i
foreach var in `apoyo' {
	bys folio: egen hapoyo_`var'=max(apoyo_`var')
}

* Genera suma que permite identificar carencia
egen c_apoyo=rsum(hapoyo_a hapoyo_c hapoyo_d hapoyo_e hapoyo_f hapoyo_g hapoyo_h hapoyo_i)
egen c_apoyo_miss=rmiss(hapoyo_a hapoyo_c hapoyo_d hapoyo_e hapoyo_f hapoyo_g hapoyo_h hapoyo_i)
replace c_apoyo=. if c_apoyo_miss>=1 & c_apoyo==0
label variable c_apoyo "Indicador apoyos sociales"
label variable c_apoyo_miss "Contador de variables perdidas en apoyos"


* Privación si para todas las situaciones no cuenta con alguna persona a fuera del hogar.
gen d_hapoyo=.
replace d_hapoyo=1 if c_apoyo==0 & pob_hapoyo==1 & c_apoyo!=. & c_apoyo_miss==0
replace d_hapoyo=0 if (c_apoyo>=1) & pob_hapoyo==1 & c_apoyo!=.
replace d_hapoyo=9 if d_hapoyo==. & pob_hapoyo==1

********************
/* 
Participación social
*/
********************

*Población de referencia
g pob_part=(edad>=14)

*Participación en organizaciones
*Organizaciones sociales
g p_orgs=.
replace p_orgs=1 if (r6<=14) & pob_part==1
replace p_orgs=0 if (r6==15) & pob_part==1

*Organizaciones del trabajo
gen p_orgt = 0 if activ == 1 & edad >= 18
replace p_orgt =. if ((o26a==-88 & o26b==-88 & o26c==-88 & o26d==-88) & activ==1 & edad>=18)
replace p_orgt = . if ((o26a == -88 | o26a == . | o26a == 2) & (o26b == -88 | o26b == . | o26b == 2) & (o26c == -88 | o26c == . | o26c == 2) & (o26d == -88 | o26d == . | o26d == 2) & activ==1 & edad>=18)

replace p_orgt = 0 if activ == 1 & edad >= 18 & (o26a==2 | o26a == .) & (o26b==2 | o26b == .) & (o26c==2  | o26c == .) & (o26d==2 | o26d == .)
replace p_orgt = 1 if activ == 1 & edad >= 18 & (o26a==1| o26b==1 | o26c==1 | o26d==1)

label define orglbl 0 "No participa" 1 "Si participa"
label values p_orgs orglbl
label values p_orgt orglbl

*Ambas organizaciones.
g d_part=.

*Para población mayor de 14 años y menor de 18 años, sólo pueden participar en organizaciones sociales
replace d_part=1 if p_orgs==0 & pob_part==1 & edad<18
replace d_part=0 if p_orgs==1 & pob_part==1 & edad<18
replace d_part=9 if p_orgs==. & pob_part==1 & edad<18

*Población mayor de 18 años
*Población mayor de 18 años e INACTIVA, sólo pueden participar en organizaciones sociales
replace d_part=1 if p_orgs==0 & (activ==2|activ==3) & edad>=18
replace d_part=0 if p_orgs==1 & (activ==2|activ==3) & edad>=18
replace d_part=9 if p_orgs==. & (activ==2|activ==3) & edad>=18

*Para población mayor de 18 años y ACTIVA, pueden participar en organizaciones sociales Y organizaciones laborales
replace d_part=1 if (p_orgs==0 & p_orgt==0) & activ==1 & pob_part==1 & edad>=18
replace d_part=0 if (p_orgs==1 | p_orgt==1) & activ==1 & pob_part==1 & edad>=18
replace d_part=9 if ((p_orgs==0 & p_orgt==.)|(p_orgs==. & p_orgt==.) | (p_orgs==. & p_orgt==0)) & activ==1 & pob_part==1 & edad>=18
label define d_part 0 "No carente" 1 "Carente"
label value d_part d_part

********************
/* 
Cálculo de carencia
*/
********************

*Población de referencia
g pob_appart=(numper!=.)

*Privación a nivel individual
g d_appart=.
replace d_appart=1 if (d_hapoyo==1) & edad<14
replace d_appart=0 if (d_hapoyo==0) & edad<14
replace d_appart=1 if (d_part==1 & d_hapoyo==1) & pob_appart==1
replace d_appart=0 if (d_part==0 | d_hapoyo==0) & pob_appart==1
replace d_appart=9 if d_appart==. & (d_hapoyo==9 | d_part==9) & pob_appart==1


********************************************************************************
/* Indicador 14: TRATO SOCIAL

Se considera carentes en entorno a hogares que declaran que alguno de sus 
miembros ha sido tratado injustamente o discriminado fuera del hogar, en los 
últimos 12 meses, por alguna de las siguientes razones: 

a. nivel socioeconómico.
b. por su nivel educacional.
c. ser hombre/mujer.
d. su estado civil.
e. su ropa.
f. su color de piel.
g. ser extranjero.
h. su edad.
i. su orientación sexual o identidad de género. 
j. tener tatuajes, piercing, perforaciones o expansiones. 
k. su apariencia física.
l. sus creencias o religión. 
m. su ideología u opinión política. 
n. participar o no en sindicatos u organizaciones gremiales. 
o. el lugar donde vive.
p. el establecimiento donde estudió. 
q. pertenecer a un pueblo indígena.
r. su condición de salud o discapacidad.

*/
********************************************************************************

*Población de referencia:
gen pob_tsocial=(numper!=.)
replace pob_tsocial=8 if numper==.

* Privación a nivel individual
gen d_tsocial=.
replace d_tsocial=1 if ( r9a==1 | r9b==1 | r9c==1 | r9d==1 | r9e==1 | r9f==1 | r9g==1 | r9h==1 | r9i==1 | r9j==1 | r9k==1 | r9l==1 | r9m==1 | r9n==1 | r9o==1 | r9p==1 | r9q==1 | r9r==1 ) & pob_tsocial==1

replace d_tsocial=0 if (r9s==1) & d_tsocial==. & pob_tsocial==1
replace d_tsocial=0 if (r9t==1) & d_tsocial==. & pob_tsocial==1


********************************************************************************
/* Indicador 15: SEGURIDAD

Un hogar se considera carente si declara haber presenciado durante el último 
mes a lo menos una de las siguientes situaciones con frecuencia “siempre”: 

i) Tráfico de drogas
ii) Balaceras o disparos

*/
********************************************************************************

g trafico=.
replace trafico=1 if v36c==4
replace trafico=0 if (v36c==1 | v36c==2 | v36c==3)

g balacera=.
replace balacera=1 if v36e==4
replace balacera=0 if (v36e==1 | v36e==2 | v36e==3)
label define seguridad 0 "Nunca, Pocas o Muchas veces" 1 "Siempre"
label values trafico seguridad
label values balacera seguridad

*Población de referencia
gen pob_seg=(numper!=.)
replace pob_seg=8 if pob_seg==.

*Indicador (alguna de las situaciones críticas ocurre siempre)
gen d_seg=.
replace d_seg=0 if pob_seg==1
replace d_seg=1 if (trafico==1 | balacera==1) & pob_seg==1
replace d_seg=9 if d_seg==. & pob_seg==1

}


********************************************************************************
*******************AGREGACIÓN A NIVEL DE HOGARES********************************
{/*****************************************************************************/

rename (agua dis_agua elim_excretas) (d_agua d_dis_agua d_elim_excretas)

local miss asis rez esc des sob mal prevs acc_a acc_b acc act cot jub hacina estado habitab agua dis_agua elim_excretas servbas medio equipo tiempo accesi entorno hapoyo part appart tsocial seg 

foreach var in `miss' {
	g pob_`var'_c=pob_`var'
	replace pob_`var'_c=0 if (pob_`var'_c==8 | pob_`var'_c==.)
	g d_`var'_c=d_`var'
	replace d_`var'_c=. if (d_`var'_c==9 | d_`var'_c==.)
}

local pob asis rez esc des sob mal prevs acc_a acc_b acc act cot jub hacina estado habitab agua dis_agua elim_excretas servbas medio equipo tiempo accesi entorno hapoyo part tsocial seg

foreach var in `pob' {
	bys folio: egen hh_pob_`var'=max(pob_`var'_c)
	bys folio: egen hh_d_`var'=max(d_`var'_c)
	replace hh_d_`var'=0 if hh_pob_`var'==0
	
	label var d_`var'_c "Individuos carentes en `var' de población relevante"
}

local pob appart
foreach var in `pob' {
	bys folio: egen hh_pob_`var'=max(pob_`var'_c)
	bys folio: egen hh_d_`var'=min(d_`var'_c) // se usa min ya que la carencia a nivel de hogar es si todos los individuos son carentes
	replace hh_d_`var'=0 if hh_pob_`var'==0
	label var d_`var'_c "Individuos carentes en `var' de población relevante"
}

label var hh_d_asis "hogar carente en asistencia"
label var hh_d_rez "hogar carente en rezago escolar"
label var hh_d_esc "hogar carente en escolaridad"
label var hh_d_mal "hogar carente en malnutrición en niños/as"
label var hh_d_prevs "hogar carente en adscripción a sistema de salud"
label var hh_d_acc "hogar carente en atención"
label var hh_d_act "hogar carente en ocupación"
label var hh_d_cot "hogar carente en seguridad social"
label var hh_d_jub "hogar carente en jubilaciones"
label var hh_d_habitab "hogar carente en habitabilidad"
label var hh_d_hacina "hogar carente en hacinamiento"
label var hh_d_estado "hogar carente en estado de la vivienda"
label var hh_d_servbas "hogar carente en servicios básicos"
label var hh_d_entorno "hogar carente en entorno"
label var hh_d_accesi "hogar carente en accesibilidad"
label var hh_d_medio "hogar carente en medio ambiente"
label var hh_d_appart "hogar carente en apoyo y participación social"
label var hh_d_tsocial "hogar carente en trato igualitario"
label var hh_d_seg "hogar carente en seguridad"
	
local pob asis rez esc des sob mal prevs acc_a acc_b acc act cot jub hacina estado habitab agua dis_agua elim_excretas servbas medio equipo tiempo accesi entorno hapoyo part appart tsocial seg 
foreach var in `pob' {
	label define hh_d_`var' 0 "No carente" 1 "Carente", modify
	label values hh_d_`var' hh_d_`var'
}

}


********************************************************************************
************************INDICADOR SINTÉTICO*************************************
{/*****************************************************************************/




********************
/* 
5 dimensiones
*/
********************

*5.1. Ponderación
*PESO DE DIMENSIONES TRADICIONALES
local pob asis rez esc mal prevs acc act cot jub habitab servbas entorno
foreach var in `pob' {
	* Peso para cada indicador
	gen pp_`var'= 0.225/3
	* Carencia ponderada
	gen w_hh_d_`var'= hh_d_`var'* pp_`var'
	*label var hh_d_`var' "Hogar carente en `var'",
	label var pp_`var' "Ponderador `var'"
label var w_hh_d_`var' "Carencia ponderada de `var'"
}


*PESO DE DIMENSIÓN DE COHESIÓN SOCIAL
local pob appart tsocial seg
foreach var in `pob' {
	* Peso para cada indicador
	gen pp_`var'= 0.1/3
	* Carencia ponderada
	gen w_hh_d_`var'= hh_d_`var'* pp_`var'
	label var pp_`var' "Ponderador `var'"
	label var w_hh_d_`var' "Carencia ponderada de `var'"
}


*5.2 Vector de conteo
* El siguiente comando suma todos los indicadores ponderados
egen float ci=rsum(w_hh_d_*)
label var ci "Vector de Conteo"
* Se identifican hogares excluidos por no contar con información para todos los indicadores
egen missing=rmiss(w_hh_d_*)
gen dmissing=(missing!=0)
label var missing "Conteo de Missing"

* 5.3. Identificación y Agregación
*UMBRAL K=22.5%
gen h_k=(ci>=0.225) if dmissing!=1
label var h_k "situación de pobreza multidimensional con entorno y redes (5 dimensiones)"
gen a_k=ci if h_k==1
label var a_k "Intensidad de la pobreza multidimensional con entorno y redes (5 dimensiones)"
gen m_k=0 if dmissing!=1
replace m_k=ci if h_k==1
label var m_k "Tasa de recuento ajustado de pobreza multidimensional con entorno y redes (5 dimensiones)"


rename h_k pobreza_multi_5d
label define pobreza_multi_5d 0 "No pobre" 1 "Pobre",modify
label values pobreza_multi_5d pobreza_multi_5d



********************
/* 
4 Dimensiones
*/
********************

*PESO DE DIMENSIONES TRADICIONALES
local pob asis rez esc mal prevs acc act cot jub hacina estado servbas
foreach var in `pob' {
	* Peso para cada indicador
	gen pp_`var'_ant= 0.25/3
	* Carencia ponderada
	gen w_hh_d_`var'_ant= hh_d_`var'* pp_`var'_ant
	label var pp_`var'_ant "Ponderador `var'"
	label var w_hh_d_`var'_ant "Carencia ponderada de `var'"
}

*6.2. Vector de conteo
* El siguiente comando suma todos los indicadores ponderados
egen float ci_ant=rsum(w_hh_d_*_ant)
label var ci_ant "Vector de Conteo"
* Se identifican hogares excluidos por no contar con información para todos los indicadores
egen missing_ant=rmiss(w_hh_d_*_ant)
gen dmissing_ant=(missing_ant!=0)
label var missing_ant "Conteo de Missing"

*6.3 Identificación y Agregación

*UMBRAL K=25%
gen h_ant=(ci_ant>=0.25) if dmissing_ant!=1
label var h_ant "situación de pobreza multidimensional (4 dimensiones)"
gen a_ant=ci_ant if h_ant==1
label var a_ant "Intensidad de pobreza multidimensional (4 dimensiones)"
gen m_ant=0 if dmissing_ant!=1
replace m_ant=ci_ant if h_ant==1
label var m_ant "Tasa de recuento ajustado de pobreza multidimensional (4 dimensiones)"

*Se cambia nombre de la variable de incidencia para cada una de las medidas
rename h_ant pobreza_multi_4d
label define pobreza_multi_4d 0 "No pobre" 1 "Pobre", modify
label values pobreza_multi_4d pobreza_multi_4d

}


*Estimaciones de la incidencia de 4 y 5 dimensiones a nivel de individuo y hogares

*Definimos el diseño complejo de la encuesta
svyset varunit [w=expr], strata(varstrat) psu(varunit) singleunit(certainty)

*Estimación a nivel de individuo:
svy: prop pobreza_multi_4d
svy: prop pobreza_multi_5d

*Estimación a nivel de hogar:
svy, subpop(if pco1 == 1): prop pobreza_multi_4d
svy, subpop(if pco1 == 1): prop pobreza_multi_5d

